﻿//-----------------------------------------------------------------------------
// PlatformerGame.js
//
// Inspired by the Microsoft XNA Community Game Platformer Sample
// Copyright (C) Microsoft Corporation. All rights reserved.
// Ported to HTML5 Canvas with EaselJS by David Rousset - http://blogs.msdn.com/davrous
//-----------------------------------------------------------------------------

/// <summary>
/// This is the main type for your game
/// </summary>
(function (window) {
    //usefull keycodes
    var KEYCODE_SPACE = 32;
    var KEYCODE_UP = 38;
    var KEYCODE_LEFT = 37;
    var KEYCODE_RIGHT = 39;
    var KEYCODE_W = 87;
    var KEYCODE_A = 65;
    var KEYCODE_D = 68;

    // The number of levels in the Levels directory of our content. We assume that
    // levels in our content are 0-based and that all numbers under this constant
    // have a level file present. This allows us to not need to check for the file
    // or handle exceptions, both of which can add unnecessary time to level loading.
    var numberOfLevels = 4;

    var numberOfLevelDownloaded = 0;

    // Used in case of an HTTP issue or access denied on file://
    // This is a static level. So if you're looping always on the same level
    // You're stuck in the Matrix because of an exception somewhere... Up to you to find where!
    var hardcodedErrorTextLevel = ".....................................................................................................................................................GGG.................###................................GGG.......GGG.......###...--..###........................1................X.####################";

    // Variables used to handle the overlay to display "You died", "You win", etc.
    var overlayDisplayed = false;
    var statusBitmap = null;
    var scoreText = null;
    var timeRemainingText = null;

    // Displaying the timer in red under 30s of remaining time
    var WarningTime = 30;

    function PlatformerGame(stage, contentManager, gameWidth, gameHeight, displayWidth, displayHeight) {
        this.platformerGameStage = stage;
        this.platformerGameContentManager = contentManager;
        this.gameWidth = gameWidth;
        this.gameHeight = gameHeight;
        this.displayWidth = displayWidth;
        this.displayHeight = displayHeight;
        this.levelIndex = -1;
        this.level = null;
        this.wasContinuePressed = false;
        this.continuePressed = false;
        this.loadNextLevel = false;

        // Little closure needed here
        var instance = this; // store the current context

        // Our hero can be moved with the arrow keys (left, right)
        // And jump with W
        document.onkeydown = function (e) {
            instance.handleKeyDown(e);
        };

        document.onkeyup = function (e) {
            instance.handleKeyUp(e);
        };

        this.registerTransitionEndEvents();

        PlatformerGame.IsOnline = this.CheckIfOnline();

        // If we're online, we're downloading/updating all the levels
        // from the webserver
        if (PlatformerGame.IsOnline) {
            this.DownloadAllLevels();
        }
        // If we're offline, we're loading the first level
        // from the cache handled by the local storage
        else {
            this.LoadNextLevel();
        }
    };

    // Registering to the various browsers vendors transition end event
    PlatformerGame.prototype.registerTransitionEndEvents = function () {
        // IE10, Firefox, Chrome & Safari, Opera
        this.platformerGameStage.canvas.addEventListener("MSTransitionEnd", onTransitionEnd(this));
        this.platformerGameStage.canvas.addEventListener("transitionend", onTransitionEnd(this));
        this.platformerGameStage.canvas.addEventListener("webkitTransitionEnd", onTransitionEnd(this));
        this.platformerGameStage.canvas.addEventListener("oTransitionEnd", onTransitionEnd(this));
    };

    // Function called when the transition has ended
    // We're then loading the next level
    function onTransitionEnd(instance) {
        return function () {
            if (instance.loadNextLevel === true) {
                instance.LoadNextLevel();
            }
        }
    };

    // Update logic callbacked by EaselJS
    // Equivalent of the Update() method of XNA
    PlatformerGame.prototype.tick = function () {
        try {
            if (this.level !== null) {
                this.HandleInput();
                this.level.Update();
                this.UpdateScore();

                // If the hero died or won, display the appropriate overlay
                if (!overlayDisplayed) {
                    this.DrawOverlay();
                }
            }
        }
        catch (e) {
            //console.log('Error', e);
        }
    };

    // Starting the game
    PlatformerGame.prototype.StartGame = function () {
        // we want to do some work before we update the canvas,
        // otherwise we could use Ticker.addListener(stage);
        Ticker.addListener(this);
        // Targeting 60 FPS
        Ticker.useRAF = enableRAF;
        Ticker.setFPS(60);
    };

    // Well, the method's name should be self explicit ;-)
    PlatformerGame.prototype.UpdateScore = function () {
        if (scoreText === null) {
            timeRemainingText = new Text("TIME: ", "bold 14px Arial", "yellow");
            timeRemainingText.x = 10;
            timeRemainingText.y = 20;
            this.platformerGameStage.addChild(timeRemainingText);

            scoreText = new Text("SCORE: 0", "bold 14px Arial", "yellow");
            scoreText.x = 10;
            scoreText.y = 34;
            this.platformerGameStage.addChild(scoreText);
        }

        if (this.level.TimeRemaining < WarningTime && !this.level.ReachedExit) {
            timeRemainingText.color = "red";
        }
        else {
            timeRemainingText.color = "yellow";
        }

        scoreText.text = "SCORE: " + this.level.Score;
        timeRemainingText.text = "TIME: " + parseInt(this.level.TimeRemaining);
    };

    // Perform the appropriate action to advance the game and
    // to get the player back to playing.
    PlatformerGame.prototype.HandleInput = function () {
        if (!this.wasContinuePressed && this.continuePressed) {
            if (!this.level.Hero.IsAlive) {
                this.level.StartNewLife();
            }
            else if (this.level.TimeRemaining == 0) {
                if (this.level.ReachedExit) {
                    // If CSS3 Transitions is supported
                    // We're using smooth & nice effects between each levels
                    if (Modernizr.csstransitions) {
                        this.loadNextLevel = true;
                        // Setting the moveRotation class will trigger the css transition
                        this.platformerGameStage.canvas.className = "moveRotation";
                    }
                    // If CSS3 Transition is not supported, we're jumping directly
                    // to the next level
                    else {
                        this.LoadNextLevel();
                    }
                }
                else
                    this.ReloadCurrentLevel();
            }
            this.platformerGameStage.removeChild(statusBitmap);
            overlayDisplayed = false;
        }

        this.wasContinuePressed = this.continuePressed;
    };

    // Determine the status overlay message to show.
    PlatformerGame.prototype.DrawOverlay = function () {
        var status = null;

        if (this.level.TimeRemaining == 0) {
            if (this.level.ReachedExit) {
                status = this.platformerGameContentManager.winOverlay;
            }
            else {
                status = this.platformerGameContentManager.loseOverlay;
            }
        }
        else if (!this.level.Hero.IsAlive) {
            status = this.platformerGameContentManager.diedOverlay;
        }

        if (status !== null) {
            statusBitmap = new Bitmap(status);
            statusBitmap.x = (this.gameWidth - statusBitmap.image.width) / 2;
            statusBitmap.y = (this.gameHeight - statusBitmap.image.height) / 2;
            overlayDisplayed = true;
            this.platformerGameStage.addChild(statusBitmap);
        }
    };

    PlatformerGame.prototype.DownloadAllLevels = function () {
        // Searching where we are currently hosted
        var levelsUrl = window.location.href.replace('index.html', '') + "levels/";
        var that = this;

        for (var i = 0; i < numberOfLevels; i++) {
            try {
                var request = new XMLHttpRequest();
                request.open('GET', levelsUrl + i + ".txt", true);
                request.onreadystatechange = makeStoreCallback(i, request, that);
                request.send(null);
            }
            catch (e) {
                // Probably an access denied if you try to run from the file:// context
                // Loading the hard coded error level to have at least something to play with
                //console.log("Error in XHR. Are you offline?"); 
                if (!window.localStorage["platformer_level_0"]) {
                    window.localStorage["platformer_level_0"] = hardcodedErrorTextLevel;
                }
            }
        }
    };

    // Closure of the index 
    function makeStoreCallback(index, request, that) {
        return function () {
            storeLevel(index, request, that);
        }
    }

    function storeLevel(index, request, that) {
        if (request.readyState == 4) {
            // If everything was OK
            if (request.status == 200) {
                // storing the level in the local storage
                // with the key "platformer_level_{index}
                window.localStorage["platformer_level_" + index] = request.responseText.replace(/[\n\r\t]/g, '');
                numberOfLevelDownloaded++;
            }
            else {
                // Loading a hard coded level in case of error
                window.localStorage["platformer_level_" + index] = hardcodedErrorTextLevel;
            }

            if (numberOfLevelDownloaded === numberOfLevels) {
                that.LoadNextLevel();
            }
        }
    }

    PlatformerGame.prototype.CheckIfOnline = function () {
        if (!navigator.onLine) return false;

        var levelUrl = window.location.href.replace('index.html', '') + "levels/" + "0.txt";

        try {
            var request = new XMLHttpRequest();
            request.open('GET', levelUrl, false);
            request.send(null);
        }
        catch (e) {
            return false;
        }

        if (request.status !== 200) {
            return false;
        }
        else {
            return true;
        }
    };

    // Loading the next level contained into the localStorage in platformer_level_{index}
    PlatformerGame.prototype.LoadNextLevel = function () {
        this.loadNextLevel = false;
        // Setting back the initialRotation class will trigger the transition
        this.platformerGameStage.canvas.className = "initialRotation";
        this.levelIndex = (this.levelIndex + 1) % numberOfLevels;
        var newTextLevel = window.localStorage["platformer_level_" + this.levelIndex];
        this.LoadThisTextLevel(newTextLevel);
    };


    PlatformerGame.prototype.LoadThisTextLevel = function (textLevel) {
        scoreText = null;

        // Unloads the content for the current level before loading the next one.
        if (this.level != null)
            this.level.Dispose();

        this.level = new Level(this.platformerGameStage, this.platformerGameContentManager, textLevel, this.gameWidth, this.gameHeight);
        this.level.StartLevel();
    };

    // Loaded if the hero lost because of a timeout
    PlatformerGame.prototype.ReloadCurrentLevel = function () {
        --this.levelIndex;
        this.LoadNextLevel();
    };

    PlatformerGame.prototype.handleKeyDown = function (e) {
        //cross browser issues exist
        if (!e) { var e = window.event; }
        switch (e.keyCode) {
            case KEYCODE_A: ;
            case KEYCODE_LEFT:
                this.level.Hero.direction = -1;
                break;
            case KEYCODE_D: ;
            case KEYCODE_RIGHT:
                this.level.Hero.direction = 1;
                break;
            case KEYCODE_W:
                this.level.Hero.isJumping = true;
                this.continuePressed = true;
        }
    };

    PlatformerGame.prototype.handleKeyUp = function (e) {
        //cross browser issues exist
        if (!e) { var e = window.event; }
        switch (e.keyCode) {
            case KEYCODE_A: ;
            case KEYCODE_LEFT: ;
            case KEYCODE_D: ;
            case KEYCODE_RIGHT:
                this.level.Hero.direction = 0;
                break;
            case KEYCODE_W:
                this.continuePressed = false;
                break;
        }
    };

    window.PlatformerGame = PlatformerGame;
} (window));